/*
 * Deformable Landscape Program
 * ----------------------------
 *
 * Copyright (C) 2000 by Marc Ebner
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * This program may be freely copied, modified and redistributed without
 * fee for non-commercial purposes provided that this copyright notice is
 * preserved intact on all copies and modified copies.
 *
 * This program accompanies the paper "Co-evolutionary dynamics on a
 * deformable landscape" by Marc Ebner, Richard A. Watson, and Jason Alexander.
 * In Proceedings of the 2000 Congress on Evolutionary Computation,
 * 16-19 July 2000, La Jolla Marriott Hotel, San Diego, USA.
 *
 * Marc Ebner
 * Universitaet Wuerzburg
 * Lehrstuhl fuer Informatik II
 * Am Hubland
 * 97074 Wuerzburg
 * Germany
 * E-Mail: ebner@informatik.uni-wuerzburg.de
 * or      m.ebner.1@alumni.nyu.edu
 *
 * Richard A. Watson
 * Brandeis University
 * Volen Center for Complex Systems
 * Mail Stop 18
 * Waltham, MA 02454-9110
 * USA
 * E-Mail: richardw@cs.brandeis.edu
 *
 * Jason Alexander
 * University of California, Irvine
 * Logic & Philosophy of Science
 * School of Social Science
 * Irvine, CA 92697
 * USA
 * E-Mail: jalex@uci.edu
 *
 */

#include <math.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "ps.h"

/*
 * definitions
 * -----------
 */

#define LANDSCAPE_WIDTH 1001
#define LANDSCAPE_HEIGHT -5
#define GAUSSIAN_WIDTH 1001
#define HALFGAUSSIAN_WIDTH 500
#define MAX_IND 100
#define MAX_HIST 100

#define MAX_NSHAPES 3
#define MAX_NINDINT 2
#define MAX_NENVINT 2

#define OUT_VIEW 1
#define OUT_PS   2

#define MAX_UDV 3
#define CURSOR_Y 26
#define LETTER_Y 7

/*
 * structure for individuals
 * -------------------------
 */

struct IndStruct {
  float x;
  float y; 
  float noise;
  float v;
  float (*updateVelocity)(float,float,float *,int);
  int   l;
  int   i;
  float stddev;
  float xOld[MAX_HIST];
  float yOld[MAX_HIST];
  float *gaussian;
};

typedef struct IndStruct IndData;

/*
 * structure for environment
 * -------------------------
 */

struct EnvStruct {
  FILE  *gpFile;
  IndData   individuals[MAX_IND];
  float landscape[LANDSCAPE_WIDTH];
  float indLandscape[LANDSCAPE_WIDTH];
  float staticLandscape[LANDSCAPE_WIDTH];
  int   width;
  float top;
  float bottom;
  int   n;
  int   hills;
  float sigma;
  int   updateVelocity;
  float noise;
  int   latency;
  int   output;
  int   shape;
  int   indInt;
  int   envInt;
  int   startSequence;
  int   running;
  int   started;

  char  filename[256];
  int   landscapeNr;
  int   frameSkip;
  int   frameSkipCounter;
};

typedef struct EnvStruct EnvData;

/*
 * structure for different choices
 * -------------------------------
 */

struct ChoiceStruct {
  char *text;
  void *f;
};

typedef struct ChoiceStruct ChoiceData;

/*
 * structure for menu
 * ------------------
 */

struct MenuStruct {
  char  *text;
  int   n;
  int   width;
  float sigma;
  int   updateVelocity;
  float noise;
  int   latency;
  int   indInt;
  int   envInt;
  int   shape;
  int   hills;
};

typedef struct MenuStruct MenuData;

/*
 * global variables
 * ----------------
 */

float hat[GAUSSIAN_WIDTH];
float dgaussian[GAUSSIAN_WIDTH];
float gaussian[GAUSSIAN_WIDTH];

float envIntLinear(float land,float ind);
float envIntComplex(float land,float ind);
float indIntLinear(float land,float ind);
float indIntComplex(float land,float ind);
float velocityConstant(float x,float v,float *land,int width);
float velocityProportional(float x,float v,float *land,int width);
float velocityIntegratedDamped(float x,float v,float *land,int width);
float velocityIntegratedUndamped(float x,float v,float *land,int width);

ChoiceData indIntChoice[MAX_NINDINT]={
    {"Linear", (void *)indIntLinear},
    {"Complex",(void *)indIntComplex}
};
ChoiceData envIntChoice[MAX_NENVINT]={
    {"Linear", (void *)envIntLinear},
    {"Complex",(void *)envIntComplex}
};
ChoiceData shapeChoice[MAX_NSHAPES]={
    {"Gaussian", (void *)gaussian},
    {"Hat",      (void *)hat},
    {"DGaussian",(void *)dgaussian}
};
ChoiceData updateVelocityChoice[MAX_UDV]={
    {"Constant",         (void *)velocityConstant},
    {"Proportional",     (void *)velocityProportional},
    {"Integrated Damped",(void *)velocityIntegratedDamped}
};

/*
 * get random integer with range [l,r]
 * -----------------------------------
 * parameters
 *   int l: lower bound
 *   int r: upper bound
 * returns
 *   int    random integer with range [l,r]
 */

int
getRandomInt(int l,int r)
{
  return (int) ((r-l+1.0)*drand48())+l;
}

/*
 * calculate gradient of landscape
 * -------------------------------
 * parameters
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 *   int x:       position on landscape
 * returns
 *   float:       gradient at position x
 */

float
gradient(float *land,int width,int x)
{
  int x0;
  int x1;
  int x2;

  x0=x-1;
  x1=x;
  x2=x+1;
  if (x0<0)
    x0+=width;
  if (x2>=width)
    x2-=width;
  assert(x2>=0 && x2<LANDSCAPE_WIDTH);
  assert(x1>=0 && x1<LANDSCAPE_WIDTH);
  assert(x0>=0 && x0<LANDSCAPE_WIDTH);
  return (land[x2]-land[x1] + land[x1]-land[x0])/2;
}

/*
 * velocity update method: constant velocity
 * -----------------------------------------
 * parameters
 *   float x:     position on landscape
 *   float v:     velocity of individual
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 * returns
 *   float:       new velocity
 */

float
velocityConstant(float x,float v,float *land,int width)
{
  float g=gradient(land,width,x);

  if (g<0.0)
    return -1.0;
  else if (g>0.0)
    return 1.0;

  return 0.0;
}

/*
 * velocity update method: velocity is proportional to the landscape's gradient
 * ----------------------------------------------------------------------------
 * parameters
 *   float x:     position on landscape
 *   float v:     velocity of individual
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 * returns
 *   float:       new velocity
 */

float
velocityProportional(float x,float v,float *land,int width)
{
  float alpha=10.0;

  return alpha*gradient(land,width,x);
}

/*
 * velocity update method: integrated and damped
 * ---------------------------------------------
 * parameters
 *   float x:     position on landscape
 *   float v:     velocity of individual
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 * returns
 *   float:       new velocity
 */

float
velocityIntegratedDamped(float x,float v,float *land,int width)
{
  float alpha=10.0;

  v+=alpha*gradient(land,width,x);
  v-=0.1*v;
  return v;
}

/*
 * velocity update method: integrated
 * ----------------------------------
 * parameters
 *   float x:     position on landscape
 *   float v:     velocity of individual
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 * returns
 *   float:       new velocity
 */

float
velocityIntegratedUndamped(float x,float v,float *land,int width)
{
  float alpha=10.0;

  v+=alpha*gradient(land,width,x);
  return v;
}

/*
 * individuals are moved accordint to their velocity
 * -------------------------------------------------
 * parameters
 *   IndData *individuals: pointer to individuals
 *   int n:                number of individuals
 *   float *land:          pointer to landscape
 *   int width:            width of landscape
 * returns
 *   void:                 nothing
 */

void
moveIndividuals(IndData *individuals,int n,float *land,int width)
{
  int i;
  IndData *ind;

  for (i=0;i<n;i++) {
    assert(i>=0 && i<MAX_IND);
    ind=&individuals[i];
    assert(ind->i>=0 && ind->i<MAX_HIST);
    ind->xOld[ind->i]=ind->x;
    ind->yOld[ind->i]=ind->y;
    ind->i++;
    if (ind->i>=MAX_HIST)
      ind->i=0;
    ind->x+=ind->v+ind->noise*(drand48()-0.5);
    while (ind->x>=width)
      ind->x-=width;
    while (ind->x<0)
      ind->x+=width;
    ind->v=ind->updateVelocity(ind->x,ind->v,land,width);
  }
}

/*
 * interaction between environment and individuals: linear
 * -------------------------------------------------------
 * parameters
 *   float land: height of landscape
 *   float ind:  height due to individuals
 * returns
 *   float:      resulting height of landscape
 */

float
envIntLinear(float land,float ind)
{
  return land-ind;
}

/*
 * interaction between environment and individuals: complex
 * --------------------------------------------------------
 * parameters
 *   float land: height of landscape
 *   float ind:  height due to individuals
 * returns
 *   float:      resulting height of landscape
 */

float
envIntComplex(float land,float ind)
{
  return land*ind;
}

/*
 * handle interaction between individuals and environment
 * ------------------------------------------------------
 * parameters
 *   float *indLand:    landscape due to individuals
 *   float *staticLand: static landscape
 *   float *land:       resulting landscape
 *   int width:         width of landscape
 *   int envInt:        type of interaction between individuals and environment
 * returns
 *   void:              nothing
 */

void
envInteraction(float *indLand,float *staticLand,float *land,
               int width,int envInt)
{
  int i;
  float (*system)(float,float);

  system=envIntChoice[envInt].f;
  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    land[i]=(*system)(staticLand[i],indLand[i]);
  }
}

/*
 * place individuals on landscape (after they have moved)
 * ------------------------------------------------------
 * parameters
 *   IndData *ind: pointer to individuals
 *   int n:        number of individuals
 *   float *land:  pointer to landscape
 * returns
 *   void:         nothing
 */

void
placeIndividualsOnLandscape(IndData *ind,int n,float *land)
{
  int i;

  for (i=0;i<n;i++) {
    assert(i>=0 && i<MAX_IND);
    ind[i].y=land[(int)ind[i].x];
  }
}

/*
 * interaction between individuals: linear
 * ---------------------------------------
 * parameters
 *   float land: height of landscape due to previously considered individuals
 *   float ind:  height due to individual
 * returns
 *   float:      resulting height of landscape
 */

float
indIntLinear(float land,float ind)
{
  return land+ind;
}

/*
 * interaction between individuals: complex
 * ----------------------------------------
 * parameters
 *   float land: height of landscape due to previously considered individuals
 *   float ind:  height due to individual
 * returns
 *   float:      resulting height of landscape
 */

float
indIntComplex(float land,float ind)
{
  return land*ind;
}

/*
 * handle interaction between individuals
 * --------------------------------------
 * parameters
 *   IndData *individuals: pointer to individuals
 *   int n:                number of individuals
 *   float *indLand:       pointer to landscape to due individuals
 *   int indInt:           type of interaction between individuals
 * returns
 *   void:                 nothing
 */

void
indInteraction(IndData *individuals,int n,float *indLand,int width,int indInt)
{
  IndData *ind;
  int i;
  int gx;
  int x;
  int index;
  float (*system)(float,float);

  system=indIntChoice[indInt].f;
  for (x=0;x<width;x++)
    indLand[x]=0;

  i=0;
  assert(i>=0 && i<MAX_IND);
  ind=&individuals[i];
  assert(ind->l>=0 && ind->l<MAX_HIST);
  index=ind->i-ind->l;
  if (index<0)
    index+=MAX_HIST;
  assert(index>=0 && index<MAX_HIST);
  x=(int)ind->xOld[index];
  assert(x>=0 && x<LANDSCAPE_WIDTH);
  x-=HALFGAUSSIAN_WIDTH;
  if (x<0)
    x+=width;
  for (gx=0;gx<GAUSSIAN_WIDTH;gx++) {
    if (x>=width)
      x-=width;
    indLand[x]=ind->gaussian[gx];
    x++;
  }

  for (i=1;i<n;i++) {
    ind=&individuals[i];
    index=ind->i-ind->l;
    if (index<0)
      index+=MAX_HIST;
    x=(int)ind->xOld[index];
    x-=HALFGAUSSIAN_WIDTH;
    if (x<0)
      x+=width;
    for (gx=0;gx<GAUSSIAN_WIDTH;gx++) {
      if (x>=width)
        x-=width;
      indLand[x]=(*system)(indLand[x],ind->gaussian[gx]);
      x++;
    }
  }
}

/*
 * read in landscape from file
 * ---------------------------
 * parameters
 *   char *filename: pointer to filename
 *   float *land:    pointer to landscape
 *   int width:      width of landscape
 * returns
 *   void:           nothing
 */

void
readLandFile(char *filename,float *land,int width)
{
  FILE *fd;
  int i;

  fd=fopen(filename,"r");
  assert(fd!=NULL);
  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    fscanf(fd,"%f\n",&land[i]);
  }
  fclose(fd);
}

/*
 * write landscape to file
 * -----------------------
 * parameters
 *   char *filename: pointer to filename
 *   float *land:    pointer to landscape
 *   int width:      width of landscape
 * returns
 *   void:           nothing
 */

void
writeLandFile(char *filename,float *land,int width)
{
  FILE *fd;
  int i;

  fd=fopen(filename,"w");
  assert(fd!=NULL);
  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    fprintf(fd,"%f\n",land[i]);
  }
  fclose(fd);
}

/*
 * write positions of individuals to file
 * --------------------------------------
 * parameters
 *   char *filename: pointer to filename
 *   IndData *ind:   pointer to individuals
 *   int n:          number of individuals
 * returns
 *   void:           nothing

 */

void
writeIndFile(char *filename,IndData *ind,int n)
{
  FILE *fd;
  int i;

  fd=fopen(filename,"w");
  assert(fd!=NULL);
  for (i=0;i<n;i++) {
    assert(i>=0 && i<MAX_IND);
    fprintf(fd,"%f %f\n",ind[i].x,ind[i].y);
  }
  fclose(fd);
}

/*
 * get maximum height of landscape
 * -------------------------------
 * parameters
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 * returns
 *   float:       maximum height of landscape
 */

float
getMaximum(float *land,int width)
{
  int i;
  float max;

  max=0;
  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    if (land[i]>max)
      max=land[i];
  }
  return max;
}

/*
 * copy landscape
 * --------------
 * parameters
 *   float *land:    pointer to landscape
 *   int width:      width of landscape
 *   float *srcLand: pointer to source landscape
 * returns
 *   void:           nothing
 */

void
copyLandscape(float *land,int width,float *srcLand)
{
  int i;

  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    land[i]=srcLand[i];
  }
}

/*
 * init landscape
 * --------------
 * paramerters
 *   float *land:     pointer to landscape
 *   int width:       width of landscape
 *   int hills:       number of hills
 *   float *gaussian: pointer to gaussian hill data
 * returns
 *   void:            nothing
 */

void
initLandscape(float *land,int width,int hills, float *gaussian)
{
  int i;
  int gx;
  int x;

  for (i=0;i<width;i++) {
    assert(i>=0 && i<LANDSCAPE_WIDTH);
    land[i]=0;
  }

  for (i=0;i<hills;i++) {
    x=(int)getRandomInt(0,width-1);
    assert(x>=0 && x<LANDSCAPE_WIDTH);
    x-=HALFGAUSSIAN_WIDTH;
    if (x<0)
      x+=width;
    for (gx=0;gx<GAUSSIAN_WIDTH;gx++) {
      if (x>=width)
        x-=width;
      land[x]+=gaussian[gx];
      x++;
    }
  }
}

/*
 * init one individual
 * -------------------
 * parameters
 *   IndData *ind: pointer to individuals
 *   float x:      x coordinate of individual
 *   float y:      y coordinate of individual
 *   float v:      velocity of individual
 *   float (*updateVelocity)(float,float,float *,int): update rule for velocity
 *   float noise:  noise to apply after movement
 *   float l:      latency
 *   float *shape: shape of individual
 * returns
 *   void:         nothing
 */

void
initOneIndividual(IndData *ind,float x,float y,float v,
                  float (*updateVelocity)(float,float,float *,int),
                  float noise,float l,float *shape)
{
  int i;

  ind->x=x;
  ind->y=y;
  ind->v=v;
  ind->updateVelocity=updateVelocity;
  ind->l=l;
  ind->noise=noise;
  ind->i=0;
  for (i=0;i<MAX_HIST;i++) {
    assert(i>=0 && i<MAX_HIST);
    ind->xOld[i]=x;
    ind->yOld[i]=y;
  }
  ind->gaussian=shape;
}

/*
 * init all individuals
 * --------------------
 * parameters
 *   IndData *ind: pointer to individuals
 *   int n:        number of individuals
 *   float (*updateVelocity)(float,float,float *,int): update rule for velocity
 *   float noise:  noise to apply after movement
 *   float l:      latency
 *   float *land:  pointer to landscape
 *   int width:    width of landscape
 *   float *shape: shape of individuals
 * returns
 *   void:         nothing
 */

void
initIndividuals(IndData *ind,int n,
                float (*updateVelocity)(float,float,float *,int),
                float noise,float l,float *land,int width,float *shape)
{
  int i;
  float x;

  for (i=0;i<n;i++) {
    assert(i>=0 && i<MAX_IND);
    x=getRandomInt(0,width-1);
    assert(x>=0 && i<LANDSCAPE_WIDTH);
    initOneIndividual(&ind[i],x,land[(int)x],0,updateVelocity,noise,l,shape);
  }
}

/*
 * calcuate y coordinate of landscape
 * ----------------------------------
 * parameters
 *   float *land: pointer to landscape
 *   int width:   width of landscape
 *   float x:     x coordinate on landscape
 * returns
 *   float:       y coordinate at position x
 */

float
curveY(float *land,int width,float x)
{
  int xl;
  int xr;

  if (x<0)
    x+=width;
  if (x>=width)
    x-=width;

  xl=(int)x;
  xr=(int)xl+1;
  if (xl<0)
    xl+=width;
  if (xr>=width)
    xr-=width;

  assert(xl>=0 && xl<width);
  assert(xr>=0 && xr<width);
  return land[xl]+(x-xl)*(land[xr]-land[xl]);
}

/*
 * draw velocity arrow above individual
 * ------------------------------------
 * parameters
 *   PSData *ps:    pointer to postscript handle
 *   float *land:   pointer to landscape
 *   int width:     width of landscape
 *   IndData *ind:  pointer to individual
 *   int frameSkip: number of frames which are skipped
 *   float radius:  radius of individual
 *   float b:       offset for bottom of graph
 * returns
 *   void:          nothing
 */

void
drawVelocityArrow(PSData *ps,float *land,int width,
                  IndData *ind,int frameSkip,float radius,float b)
{
  float x,oldX;
  float counter;
  float dx;
  float dy;
  float v;

  dx=0.33;
  dy=3*radius/ps->yScale;
  x=ind->x;
  v=ind->v*frameSkip;
  psSetLineWidth(ps,3.0);
  psBeginCurve(ps,x,b+curveY(land,width,x)+dy);
  if (v>0) {
    oldX=x;
    x++;
    if (x>=width) {
      x-=width;
      psEndCurve(ps);
      psBeginCurve(ps,x,b+curveY(land,width,x)+dy);
      oldX=x;
      x++;
    }
    counter=ind->x;
    while (counter<ind->x+v) {
      psContCurve(ps,
           oldX+dx,b+curveY(land,width,oldX)+gradient(land,width,oldX)*dx+dy,
           x-dx,b+curveY(land,width,x)-gradient(land,width,x)*dx+dy,
           (float)x,b+curveY(land,width,x)+dy);
      oldX=x;
      counter++;
      x++;
      if (counter>=ind->x+v)
        x+=ind->x+v-counter;
      if (x>=width) {
        x-=width;
        psEndCurve(ps);
        psBeginCurve(ps,x,b+curveY(land,width,x)+dy);
        oldX=x;
        x++;
      }
    }
    dx=(x-oldX)*dx;
    assert(dx>=0 && dx<=0.4);
    psContCurve(ps,
             oldX+dx,b+curveY(land,width,oldX)+gradient(land,width,oldX)*dx+dy,
             x-dx,b+curveY(land,width,x)-gradient(land,width,x)*dx+dy,
             (float)x,b+curveY(land,width,x)+dy);
  } else {
    oldX=x;
    x--;
    if (x<0) {
      x+=width;
      psEndCurve(ps);
      psBeginCurve(ps,x,b+curveY(land,width,x)+dy);
      oldX=x;
      x--;
    }
    counter=ind->x;
    while (counter>ind->x+v) {
      psContCurve(ps,
             oldX-dx,b+curveY(land,width,oldX)+gradient(land,width,oldX)*dx+dy,
             x+dx,b+curveY(land,width,x)-gradient(land,width,x)*dx+dy,
             (float)x,b+curveY(land,width,x)+dy);
      oldX=x;
      counter--;
      x--;
      if (counter<=ind->x+v)
        x+=ind->x+v-counter;
      if (x<0) {
        x+=width;
        psEndCurve(ps);
        psBeginCurve(ps,x,b+curveY(land,width,x)+dy);
        oldX=x;
        x--;
      }
    }
    dx=(oldX-x)*dx;
    assert(dx>=0 && dx<=0.4);
    psContCurve(ps,
           oldX-dx,b+curveY(land,width,oldX)+gradient(land,width,oldX)*dx+dy,
           x+dx,b+curveY(land,width,x)-gradient(land,width,x)*dx+dy,
           (float)x,b+curveY(land,width,x)+dy);
  }
  psEndCurve(ps);
  psSetLineWidth(ps,1.0);
}

/*
 * save landscape as postscript file
 * ---------------------------------
 * parameters
 *   EnvData *env:        pointer to environment
 *   char *filename:      pointer to filename
 *   float *land:         pointer to landscape
 *   int showIndividuals: flag which specifies if individuals are drawn too
 *   int showVelocity:    flag which specifies if velocity vectory are drawn
 * returns
 *   void:                nothing
 */

void
savePSLandscape(EnvData *env,char *filename,float *land,
                int showIndividuals,int showVelocity)
{
  float radius;
  int   i,j;
  float dx;
  PSData *ps;

  ps=psCreate(filename);
  psHead(ps,(float)env->width,(float)(env->top-env->bottom),
         1.0,30.0*(0.5+5)/(env->top-env->bottom));

  fprintf(ps->fd,"%% Individuals:                  %d\n",env->n);
  fprintf(ps->fd,"%% Width of Landscape:           %d\n",env->width);
  fprintf(ps->fd,"%% Height of Landscape (top):    %f\n",env->top);
  fprintf(ps->fd,"%% Height of Landscape (bottom): %f\n",env->bottom);
  fprintf(ps->fd,"%% Width of Gaussians:           %f\n",env->sigma);
  fprintf(ps->fd,"%% Velocity Update:              %s\n",
                 updateVelocityChoice[env->updateVelocity].text);
  fprintf(ps->fd,"%% Noise:                        %f\n",env->noise);
  fprintf(ps->fd,"%% Latency:                      %d\n",env->latency);
  fprintf(ps->fd,"%% Frame Skip:                   %d\n",env->frameSkip);
  fprintf(ps->fd,"%% Hills:                        %d\n",env->hills);

  psSetLineWidth(ps,1.0);
  psGraphBox(ps,(float)env->width,(float)(env->top-env->bottom));

  dx=0.33;
  psBeginCurve(ps,0.0,-env->bottom+land[0]);
  for (i=1;i<env->width;i++) {
    j=i-1;
    if (j<0)
      j+=env->width;
    psContCurve(ps,
                j+dx,-env->bottom+land[j]+gradient(land,env->width,j)*dx,
                i-dx,-env->bottom+land[i]-gradient(land,env->width,i)*dx,
                (float)i,-env->bottom+land[i]);
  }
  psEndCurve(ps);

  if (showIndividuals) {
    radius=3.0;
    for (i=0;i<env->n;i++) {
      psFilledCircle(ps,env->individuals[i].x,
                        -env->bottom+env->individuals[i].y,radius);

      if (showVelocity)
        drawVelocityArrow(ps,land,env->width,&env->individuals[i],
                          env->frameSkip,radius,-env->bottom);
    }
  }

  psTrailer(ps);
  psDestroy(ps);
}

/*
 * save landscape
 * --------------
 * parameters
 *   EnvData *env: pointer to environment
 * returns
 *   void:         nothing
 */

void
saveResults(EnvData *env)
{
  char filename[256];

  sprintf(filename,"%slandscape%04d.ps",env->filename,env->landscapeNr);
  env->landscapeNr++;
  savePSLandscape(env,filename,env->landscape,1,1);
}

/*
 * show landscape
 * --------------
 * parameters
 *   EnvData *env:        pointer to environment
 *   int showIndividuals: flag which specifies if individuals are shown
 * returns
 *   void:                nothing
 */

void
showResults(EnvData *env,int showIndividuals)
{
  writeLandFile("landscape.dat",env->landscape,env->width);
  if (showIndividuals) {
    writeIndFile("individuals.dat",env->individuals,env->n);
    fprintf(env->gpFile,
      "plot \"individuals.dat\" t \"\", \"landscape.dat\" t \"\" w l\n");
  } else
    fprintf(env->gpFile,
      "plot \"landscape.dat\" t \"\" w l\n");
  fflush(env->gpFile);
  usleep(50000);
}

/*
 * show menu on screen
 * -------------------
 * parameters
 *   MenuData *menu:   pointer to menu data
 *   int maxSelection: number of predefined settings
 *   EnvData *env:     pointer to environment data
 * returns
 *   void:             nothing
 */

void
showMenu(MenuData *menu,int maxSelection,EnvData *env)
{
  char  line[256];
  int   x;
  int   y;
  int   i;

  clear();
  x=0;y=0;

  mvaddstr(y,x,"Deformable Landscape Program\n");y++;

  mvaddstr(y,x,"This program accompanies the paper \"Co-evolutionary dynamics on a");y++;
  mvaddstr(y,x,"deformable landscape\" by Marc Ebner, Richard A. Watson, and Jason Alexander.");y++;
  mvaddstr(y,x,"In Proceedings of the 2000 Congress on Evolutionary Computation,");y++;
  mvaddstr(y,x,"16-19 July 2000, La Jolla Marriott Hotel, San Diego, USA.");y++;
  mvaddstr(y,x,"E-Mail: m.ebner.1@alumni.nyu.edu, richardw@cs.brandeis.edu, jalex@uci.edu");y++;

  mvaddstr(y,x,"-----------------------------------------------------------------------------");y++;
  sprintf(line,"I) Individuals:                  %d",env->n);
  mvaddstr(y,x,line);y++;
  sprintf(line,"H) Number of Hills               %d",env->hills);
  mvaddstr(y,x,line);y++;
  sprintf(line,"F) Shape of Deformation:         %s",
               shapeChoice[env->shape].text);
  mvaddstr(y,x,line);y++;
  sprintf(line,"W) Width of Landscape:           %d",env->width);
  mvaddstr(y,x,line);y++;
  sprintf(line,"T) Height of Landscape (top):    %f",env->top);
  mvaddstr(y,x,line);y++;
  sprintf(line,"B) Height of Landscape (bottom): %f",env->bottom);
  mvaddstr(y,x,line);y++;
  sprintf(line,"G) Width of Gaussians:           %f",env->sigma);
  mvaddstr(y,x,line);y++;
  sprintf(line,"V) Velocity Update:              %s",
               updateVelocityChoice[env->updateVelocity].text);
  mvaddstr(y,x,line);y++;
  sprintf(line,"N) Noise:                        %f",env->noise);
  mvaddstr(y,x,line);y++;
  sprintf(line,"L) Latency:                      %d",env->latency);
  mvaddstr(y,x,line);y++;
  sprintf(line,"C) Individual Interaction:       %s",
          indIntChoice[env->indInt].text);
  mvaddstr(y,x,line);y++;
  sprintf(line,"D) Environment Interaction:      %s",
          envIntChoice[env->envInt].text);
  mvaddstr(y,x,line);y++;
  mvaddstr(y,x,"R) Run");y++;
  mvaddstr(y,x,"S) Stop");y++;
  mvaddstr(y,x,"C) Continue");y++;
  mvaddstr(y,x,"E) End Run");y++;
  sprintf(line,"P) Save Screen with Filename %slandscape%04d.ps",
          env->filename,env->landscapeNr);
  mvaddstr(y,x,line);y++;
  sprintf(line,"X) Save Sequence of 51 Landscapes");
  mvaddstr(y,x,line);y++;
  sprintf(line,"Y) Frame Skip:                   %d",env->frameSkip);
  mvaddstr(y,x,line);y++;
  sprintf(line,"Z) Run and Save Sequence");
  mvaddstr(y,x,line);y++;
  sprintf(line,"O) Toggle Output: %s",(env->output)? "Output On":"Output Off");
  mvaddstr(y,x,line);y++;
  mvaddstr(y,x,"Q) Exit");y++;
  mvaddstr(y,x,"+) Save Landscape");y++;
  mvaddstr(y,x,"-) Load Landscape");y++;
  y++;

  sprintf(line,"    %s\n",menu[0].text);
  mvaddstr(y,x,line);
  y++;
  for (i=1;i<=maxSelection;i++) {
    sprintf(line,"%d.) %s\n",i,menu[i].text);
    mvaddstr(y,x,line);
    y++;
  }
  y++;

  move(y,x);addstr(" ");
  refresh();
}

/*
 * get integer value from user
 * ---------------------------
 * parameters
 *   int y:      y coordinate of cursor
 *   int x:      x coordinate of cursor
 *   int *value: pointer to place where value will be stored
 *   int ny:     new y coordinate of cursor
 *   int nx:     new x coordinate of cursor
 * returns
 *   void:       nothing
 */

void
getInputInt(int y,int x,int *value,int ny,int nx)
{
  char line[256];

  nodelay(stdscr,FALSE);echo();
  move(y,x);
  wgetnstr(stdscr,line,32);
  sscanf(line,"%d",value);
  sprintf(line,"                                ");
  mvaddstr(y,x,line);
  sprintf(line,"%d",*value);
  mvaddstr(y,x,line);
  nodelay(stdscr,TRUE);move(ny,nx);noecho();
}

/*
 * get float value from user
 * -------------------------
 * parameters
 *   int y:        y coordinate of cursor
 *   int x:        x coordinate of cursor
 *   float *value: pointer to place where value will be stored
 *   int ny:       new y coordinate of cursor
 *   int nx:       new x coordinate of cursor
 * returns
 *   void:         nothing
 */

void
getInputFloat(int y,int x,float *value,int ny,int nx)
{
  char line[256];

  nodelay(stdscr,FALSE);echo();
  move(y,x);
  wgetnstr(stdscr,line,32);
  sscanf(line,"%f",value);
  sprintf(line,"                                ");
  mvaddstr(y,x,line);
  sprintf(line,"%f",*value);
  mvaddstr(y,x,line);
  nodelay(stdscr,TRUE);move(ny,nx);noecho();
}

/*
 * get next selection (from a list of choices) from user
 * -----------------------------------------------------
 * parameters
 *   int y:      y coordinate of cursor
 *   int x:      x coordinate of cursor
 *   int *value: pointer to place where selection will be stored
 *   int ny:     new y coordinate of cursor
 *   int nx:     new x coordinate of cursor
 * returns
 *   void:       nothing
 */

void
getNextSelection(int y,int x,int *value,int n,int ny,int nx)
{
  (*value)++;
  if (*value==n)
    *value=0;
}

/*
 * init deformations
 * -----------------
 * parameters
 *   float sigma: width of gaussian
 * returns
 *   void:        nothing
 */

void
initShapes(float sigma)
{
  int i;
  int x;

  for (i=0;i<GAUSSIAN_WIDTH;i++) {
    assert(i>=0 && i<GAUSSIAN_WIDTH);
    x=i-HALFGAUSSIAN_WIDTH;
    gaussian[i]=exp(-sigma*x*x);
  }
  for (i=0;i<GAUSSIAN_WIDTH;i++) {
    assert(i>=0 && i<GAUSSIAN_WIDTH);
    x=i-HALFGAUSSIAN_WIDTH;
    dgaussian[i]=2*sqrt(sigma)*x*exp(-sigma*x*x);
  }
  for (i=0;i<GAUSSIAN_WIDTH;i++) {
    assert(i>=0 && i<GAUSSIAN_WIDTH);
    x=i-HALFGAUSSIAN_WIDTH;
    hat[i]=exp(-sigma*x*x)-1.5*exp(-sigma*5*x*x);
  }
}

/*
 * create environment
 * ------------------
 * parameters
 *   int width: width of environment
 *   int hills: number of hills
 * returns
 *   EnvData *: pointer to environment
 */

EnvData *
createEnvironment(int width,int hills)
{
  EnvData *env;

  env=(EnvData *)malloc(sizeof(EnvData));
  assert(env!=NULL);
  env->n=20;
  env->hills=hills;
  env->width=width;
  env->top=0;
  env->bottom=LANDSCAPE_HEIGHT;
  env->sigma=0.001;
  env->updateVelocity=0;
  env->output=1;
  env->indInt=0;
  env->envInt=0;
  env->startSequence=0;
  env->running=0;
  env->started=0;

  initShapes(env->sigma);
  initLandscape(env->staticLandscape,width,hills,gaussian);

  env->gpFile=popen("gnuplot -geometry 1000x150 -pointsize 3\n","w");
  fprintf(env->gpFile,"plot x\n");
  fprintf(env->gpFile,"set xrange[%d:%d]\n",0,env->width-1);
  fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
  fflush(env->gpFile);
  sleep(1);

  sprintf(env->filename,"land/");
  env->landscapeNr=0;
  env->frameSkip=10;
  env->frameSkipCounter=env->frameSkip;

  return env;
}

/*
 * destroy environment
 * -------------------
 * parameters
 *   EnvData *env: pointer to environment
 * returns
 *   void:         nothing
 */

void
destroyEnvironment(EnvData *env)
{
  fprintf(env->gpFile,"quit\n");
  fclose(env->gpFile);
  free(env);
}

/*
 * set parameters of environment accordig to settings in menu
 * ----------------------------------------------------------
 * parameters
 *   EnvData *env:   pointer to environment
 *   MenuData *menu: pointer to menu data
 *   int selection:  selection among predefined settings
 * returns
 *   void:           nothing
 */

void
setParameters(EnvData *env,MenuData *menu,int selection)
{
  env->n=menu[selection].n;
  env->width=menu[selection].width;
  env->sigma=menu[selection].sigma;
  env->updateVelocity=menu[selection].updateVelocity;
  env->noise=menu[selection].noise;
  env->latency=menu[selection].latency;
  env->indInt=menu[selection].indInt;
  env->envInt=menu[selection].envInt;
  env->shape=menu[selection].shape;
  env->hills=menu[selection].hills;
}

/*
 * handle user input
 * -----------------
 * parameters
 *   MenuData *menu:   pointer to menu
 *   int maxSelection: maximum number of predefined settings
 *   EnvData *env:     pointer to environment
 * returns
 *   void:             nothing
 */

void
handleInput(MenuData *menu,int maxSelection,EnvData *env)
{
  int  input;
  char c;
  int  selection;
  char filename[256];

  input=ERR;
  c=0;
  while (c!='Q') {
    input=getch();
    if (input!=ERR) {
      c=*keyname(input);
    } else
      c=0;
    if (c>='a' && c<='z') {
      c-='a';
      c+='A';
    }
    if (c=='O') {
      env->output=!env->output;
      showMenu(menu,maxSelection,env);
    }

    if (c=='I')
      getInputInt(LETTER_Y,33,&env->n,CURSOR_Y,0);
    if (c=='H') {
      getInputInt(LETTER_Y+1,33,&env->hills,CURSOR_Y,0);
      initLandscape(env->staticLandscape,env->width,env->hills,gaussian);
      copyLandscape(env->landscape,env->width,env->staticLandscape);
      env->top=getMaximum(env->staticLandscape,env->width)+0.5;
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
      showResults(env,0);
    }
    if (c=='F')
      getNextSelection(LETTER_Y+2,33,&env->shape,MAX_NSHAPES,CURSOR_Y,0);
    if (c=='W') {
      getInputInt(LETTER_Y+3,33,&env->width,CURSOR_Y,0);
      fprintf(env->gpFile,"set xrange[%d:%d]\n",0,env->width-1);
      fflush(env->gpFile);
    }
    if (c=='T') {
      getInputFloat(LETTER_Y+4,33,&env->top,CURSOR_Y,0);
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
    }
    if (c=='B') {
      getInputFloat(LETTER_Y+5,33,&env->bottom,CURSOR_Y,0);
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
    }
    if (c=='G') {
      getInputFloat(LETTER_Y+6,33,&env->sigma,CURSOR_Y,0);
      initShapes(env->sigma);
    }
    if (c=='V') {
      getNextSelection(LETTER_Y+7,33,&env->updateVelocity,MAX_UDV,LETTER_Y,0);
    }
    if (c=='N')
      getInputFloat(LETTER_Y+8,33,&env->noise,CURSOR_Y,0);
    if (c=='L'){
      getInputInt(LETTER_Y+9,33,&env->latency,CURSOR_Y,0);
    }
    if (c=='C')
      getNextSelection(LETTER_Y+10,33,&env->indInt,MAX_NINDINT,CURSOR_Y,0);
    if (c=='D')
      getNextSelection(LETTER_Y+11,33,&env->envInt,MAX_NENVINT,CURSOR_Y,0);
    if (c=='E') {
      env->running=0;
      env->started=0;
      env->startSequence=0;
    }
    if (c=='S')
      env->running=0;
    if (c=='C' && env->started)
      env->running=1;

    if (c=='P' && env->started) {
      saveResults(env);
    }
    if (env->startSequence) {
      env->frameSkipCounter--;
      if (env->frameSkipCounter==0) {
        env->frameSkipCounter=env->frameSkip;
        saveResults(env);
        showMenu(menu,maxSelection,env);
      }
      if (env->landscapeNr==51)
        env->startSequence=0;
    }
    if (c=='X' && env->started) {
      env->startSequence=1;
      env->landscapeNr=0;
      env->frameSkipCounter=1;
    }
    if (c=='Z') {
      env->startSequence=1;
      env->landscapeNr=0;
      env->frameSkipCounter=1;
      c='R';
    }
    if (c=='Y'){
      getInputInt(LETTER_Y+18,33,&env->frameSkip,CURSOR_Y,0);
    }

    if (env->running) {
      moveIndividuals(env->individuals,env->n,env->landscape,env->width);
      indInteraction(env->individuals,env->n,env->indLandscape,
                     env->width,env->indInt);
      envInteraction(env->indLandscape,env->staticLandscape,env->landscape,
                     env->width,env->envInt);
      placeIndividualsOnLandscape(env->individuals,env->n,env->landscape);
      if (env->output)
        showResults(env,1);
    }

    if (c=='R') {
      env->running=1;
      env->started=1;
      env->top=getMaximum(env->staticLandscape,env->width)+0.5;
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
      initIndividuals(env->individuals,env->n,
        (float (*)(float,float,float *,int))
        updateVelocityChoice[env->updateVelocity].f,
        env->noise,env->latency+1,env->landscape,env->width,
        (float *)shapeChoice[env->shape].f);
      indInteraction(env->individuals,env->n,env->indLandscape,env->width,
                     env->indInt);
      envInteraction(env->indLandscape,env->staticLandscape,env->landscape,
                     env->width,env->envInt);
      placeIndividualsOnLandscape(env->individuals,env->n,env->landscape);
      showResults(env,1);
    }
    if (c=='+'){
      sprintf(filename,"%sstatLandscape.dat",env->filename);
      writeLandFile(filename,env->staticLandscape,env->width);
      sprintf(filename,"%sstatLandscape.ps",env->filename);
      savePSLandscape(env,filename,env->staticLandscape,0,0);
    }
    if (c=='-'){
      sprintf(filename,"%sstatLandscape.dat",env->filename);
      readLandFile(filename,env->staticLandscape,env->width);
      copyLandscape(env->landscape,env->width,env->staticLandscape);
      env->top=getMaximum(env->staticLandscape,env->width)+0.5;
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
      showResults(env,0);
    }

    selection=c-'0';
    if (selection>=1 && selection<=maxSelection && env->started==0) {
      setParameters(env,menu,selection);
      initLandscape(env->staticLandscape,env->width,env->hills,gaussian);
      copyLandscape(env->landscape,env->width,env->staticLandscape);
      env->top=getMaximum(env->staticLandscape,env->width)+0.5;
      fprintf(env->gpFile,"set yrange[%f:%f]\n",env->bottom,env->top);
      fflush(env->gpFile);
      showResults(env,0);
    }

    if (input!=ERR)
      showMenu(menu,maxSelection,env);
  }
}

/*
 * beginning of main program
 * -------------------------
 * parameters
 *   int argc:    number of arguments
 *   char **argv: argument vector
 * returns
 *   int:         error flag
 */

int
main(int argc,char **argv)
{
  int   hills=0;
  int   width=LANDSCAPE_WIDTH;
  int   maxSelection=9;
  MenuData menu[32]={
  {"              Species  Update rule  Deformation  Latency  Hills",
   0,0,0,0,0,0,0,0,0,0},
  {"Experiment 1:    20      Constant    Gaussian       0       0",
   20,1001,0.001,0,0, 0,0,0,0, 0},
  {"Experiment 2:    20      Constant    Gaussian      50       0",
   20,1001,0.001,0,0,50,0,0,0, 0},
  {"Experiment 3:    20    Proportional  Gaussian       0       0",
   20,1001,0.001,1,0, 0,0,0,0, 0},
  {"Experiment 4:    20    Proportional  Gaussian      50       0",
   20,1001,0.001,1,0,50,0,0,0, 0},
  {"Experiment 5:    20     Integrated   Gaussian       0       0",
   20,1001,0.001,2,0, 0,0,0,0, 0},
  {"Experiment 6:    20     Integrated   Gaussian       2       0",
   20,1001,0.001,2,0, 2,0,0,0, 0},
  {"Experiment 7:    20     Integrated   Gaussian      50       0",
   20,1001,0.001,2,0,50,0,0,0, 0},
  {"Experiment 8:    20     Integrated   Gaussian       0      50",
   20,1001,0.001,2,0, 0,0,0,0,50},
  {"Experiment 9:    20     Integrated   Gaussian      50      50",
   20,1001,0.001,2,0,50,0,0,0,50}
  };
  EnvData *env;

  env=createEnvironment(width,hills);
  initscr();
  cbreak();noecho();
  nonl();intrflush(stdscr,FALSE);keypad(stdscr,TRUE);nodelay(stdscr,TRUE);
  showMenu(menu,maxSelection,env);
  handleInput(menu,maxSelection,env);
  endwin();
  destroyEnvironment(env);
  return 0;
}



