/*
 * Deformable Landscape Program: Code to create postscript output
 * --------------------------------------------------------------
 *
 * Copyright (C) 2000 by Marc Ebner
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * This program may be freely copied, modified and redistributed without
 * fee for non-commercial purposes provided that this copyright notice is
 * preserved intact on all copies and modified copies.
 *
 * This program accompanies the paper "Co-evolutionary dynamics on a
 * deformable landscape" by Marc Ebner, Richard A. Watson, and Jason Alexander.
 * In Proceedings of the 2000 Congress on Evolutionary Computation,
 * 16-19 July 2000, La Jolla Marriott Hotel, San Diego, USA.
 *
 * Marc Ebner
 * Universitaet Wuerzburg
 * Lehrstuhl fuer Informatik II
 * Am Hubland
 * 97074 Wuerzburg
 * Germany
 * E-Mail: ebner@informatik.uni-wuerzburg.de
 * or      m.ebner.1@alumni.nyu.edu
 *
 * Richard A. Watson
 * Brandeis University
 * Volen Center for Complex Systems
 * Mail Stop 18
 * Waltham, MA 02454-9110
 * USA
 * E-Mail: richardw@cs.brandeis.edu
 *
 * Jason Alexander
 * University of California, Irvine
 * Logic & Philosophy of Science
 * School of Social Science
 * Irvine, CA 92697
 * USA
 * E-Mail: jalex@uci.edu
 *
 */

#include <math.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "ps.h"

/*
 * create postscript file
 * ----------------------
 * parameters
 *   char *filename: filename
 * returns
 *   PSData *:       pointer to ps structure
 */

PSData *
psCreate(char *filename)
{
  PSData *ps;

  ps=(PSData *)malloc(sizeof(PSData));
  assert(ps!=NULL);
  ps->fd=fopen(filename,"w");
  assert(ps->fd!=NULL);
  strcpy(ps->filename,filename);
  return ps;
}

/*
 * destroy postscript structure
 * ----------------------------
 * parameters
 *   PSData *ps: pointer to ps structure
 * returns
 *   void:       nothing
 */

void
psDestroy(PSData *ps)
{
  fclose(ps->fd);
  free(ps);
}

/*
 * write encapsulated postscript header to file
 * --------------------------------------------
 * parameters
 *   PSData *ps:   pointer to ps structure
 *   float width:  width of document
 *   float height: height of document
 *   float xScale: scaling factor in x direction
 *   float yScale: scaling factor in y direction
 * returns
 *   void:         nothing
 */

void
psHead(PSData *ps,float width,float height,float xScale,float yScale)
{
  time_t t;

  ps->xScale=xScale;
  ps->yScale=yScale;

  t=time(NULL);
  fprintf(ps->fd,"%%!PS-Adobe-2.0 EPSF-2.0\n");
  fprintf(ps->fd,"%%%%Title: %s\n",ps->filename);
  fprintf(ps->fd,"%%%%Creator: deform\n");
  fprintf(ps->fd,"%%%%CreationDate: %s",ctime(&t));
  fprintf(ps->fd,"%%%%Pages: 1\n");
  fprintf(ps->fd,"%%%%BoundingBox: %f %f %f %f\n",
                 0.0,0.0,xScale*width,yScale*height);
  fprintf(ps->fd,"%%%%DocumentFonts: (atend)\n");
  fprintf(ps->fd,"%%%%Orientation: Portrait\n");
  fprintf(ps->fd,"%%%%EndComments\n");
}

/*
 * write postscript trailer to file
 * --------------------------------
 * parameters
 *   PSData *ps: pointer to ps structure
 * returns
 *   void:       nothing
 */

void
psTrailer(PSData *ps)
{
  fprintf(ps->fd,"showpage\n");
  fprintf(ps->fd,"%%%%Trailer\n");
  fprintf(ps->fd,"%%%%DocumentFonts: Helvetica\n");
}

/*
 * begin curve
 * -----------
 * parameters
 *   PSData *ps: pointer to ps structure
 *   float x0:   first point of curve x coordinate
 *   float y0:   first point of curve y coordinate
 * returns
 *   void:       nothing
 */

void
psBeginCurve(PSData *ps,float x0,float y0)
{
  fprintf(ps->fd,"newpath %.2f %.2f moveto\n",
          ps->xScale*x0,ps->yScale*y0);
}

/*
 * continue curve
 * --------------
 * parameters
 *   PSData *ps: pointer to ps structure
 *   float x1:   point in direction of tangent of previous point
 *   float y1:   point in direction of tangent of previous point
 *   float x2:   point in direction of tangent of next point
 *   float y2:   point in direction of tangent of next point
 *   float x1:   next point
 *   float y1:   next point
 * returns
 *   void:       nothing
 */

void
psContCurve(PSData *ps,float x1,float y1,float x2,float y2,float x3,float y3)
{
  fprintf(ps->fd,"%.2f %.2f %.2f %.2f %.2f %.2f curveto\n",
          ps->xScale*x1,ps->yScale*y1,
          ps->xScale*x2,ps->yScale*y2,
          ps->xScale*x3,ps->yScale*y3);
}

/*
 * end curve
 * ---------
 * parameters
 *   PSData *ps: pointer to ps structure
 * returns
 *   void:       nothing
 */

void
psEndCurve(PSData *ps)
{
  fprintf(ps->fd,"stroke\n");
}

/*
 * set line width
 * --------------
 * parameters
 *   PSData *ps:      pointer to ps structure
 *   float lineWidth: new line width
 * returns
 *   void:       nothing
 */

void
psSetLineWidth(PSData *ps,float lineWidth)
{
  fprintf(ps->fd,"%.2f setlinewidth\n",lineWidth);
}

/*
 * draw line
 * ---------
 * parameters
 *   PSData *ps: pointer to ps structure
 *   float x0:   x coordinate of first point
 *   float y0:   y coordinate of first point
 *   float x1:   x coordinate of second point
 *   float y1:   y coordinate of second point
 * returns
 *   void:       nothing
 */

void
psLine(PSData *ps,float x0,float y0,float x1,float y1)
{
  fprintf(ps->fd,"newpath %.2f %.2f moveto ",
          ps->xScale*x0,ps->yScale*y0);
  fprintf(ps->fd,"%f %f lineto stroke\n",
          ps->xScale*x1,ps->yScale*y1);
}

/*
 * draw filled circle
 * ------------------
 * parameters
 *   PSData *ps: pointer to ps structure
 *   float x:    x coordinate of circle
 *   float y:    y coordinate of circle
 *   float r:    radius of circle
 * returns
 *   void:       nothing
 */

void
psFilledCircle(PSData *ps,float x,float y,float r)
{
  fprintf(ps->fd,"0 setgray\n");
  fprintf(ps->fd,"newpath %f %f %f 0 360 arc fill\n",
          x*ps->xScale,y*ps->yScale,r*ps->xScale);
  fprintf(ps->fd,"0 setgray\n");
  fprintf(ps->fd,"newpath %f %f %f 0 360 arc stroke\n",
          x*ps->xScale,y*ps->yScale,r*ps->xScale);
}

/*
 * draw box around graph
 * ---------------------
 * parameters
 *   PSData *ps:   pointer to ps structure
 *   float width:  width of box
 *   float height: height of box
 * returns
 *   void:       nothing
 */

void
psGraphBox(PSData *ps,float width,float height)
{
  psLine(ps,0.0,0.0,width,0.0);
  psLine(ps,width,0.0,width,height);
  psLine(ps,width,height,0.0,height);
  psLine(ps,0.0,height,0.0,0.0);
  fprintf(ps->fd,"%f %f %f %f rectclip\n",
          0.0,0.0,ps->xScale*width,ps->yScale*height);
}

