/*
 * 2D Deformable Landscape Program
 * -------------------------------
 *
 * Copyright (C) 2000 by Marc Ebner
 *
 * Marc Ebner
 * Universitaet Wuerzburg
 * Lehrstuhl fuer Informatik II
 * Am Hubland
 * 97074 Wuerzburg
 * Germany
 * E-Mail: ebner@informatik.uni-wuerzburg.de
 * or      m.ebner.1@alumni.nyu.edu
 *
 * Portions of this code are based on the one dimensional deformable landscape
 * program:
 *
 * Deformable Landscape Program
 * ----------------------------
 *
 * Copyright (C) 2000 by Marc Ebner
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * This program may be freely copied, modified and redistributed without
 * fee for non-commercial purposes provided that this copyright notice is
 * preserved intact on all copies and modified copies.
 *
 * This program accompanies the paper "Co-evolutionary dynamics on a
 * deformable landscape" by Marc Ebner, Richard A. Watson, and Jason Alexander.
 * In Proceedings of the 2000 Congress on Evolutionary Computation,
 * 16-19 July 2000, La Jolla Marriott Hotel, San Diego, USA.
 *
 * Marc Ebner
 * Universitaet Wuerzburg
 * Lehrstuhl fuer Informatik II
 * Am Hubland
 * 97074 Wuerzburg
 * Germany
 * E-Mail: ebner@informatik.uni-wuerzburg.de
 * or      m.ebner.1@alumni.nyu.edu
 *
 * Richard A. Watson
 * Brandeis University
 * Volen Center for Complex Systems
 * Mail Stop 18
 * Waltham, MA 02454-9110
 * USA
 * E-Mail: richardw@cs.brandeis.edu
 *
 * Jason Alexander
 * University of California, Irvine
 * Logic & Philosophy of Science
 * School of Social Science
 * Irvine, CA 92697
 * USA
 * E-Mail: jalex@uci.edu
 *
 */

#include <math.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <curses.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <GL/glut.h>

/*
 * definitions
 * -----------
 */

#define LANDSCAPE_XSIZE 1001
#define LANDSCAPE_YSIZE 1001
#define MAX_IND 1000
#define MAX_HIST 100

#define MAX_NSHAPES 3
#define MAX_NINDDIS 3
#define MAX_UDV 3

/*
 * structure for individuals
 * -------------------------
 */

struct IndStruct {
  float x;
  float y; 
  float z; 
  float noise;
  float vx;
  float vy;
  int   updateVelocity;
  int   l;
  int   i;
  float xOld[MAX_HIST];
  float yOld[MAX_HIST];
  float **deform;
};

typedef struct IndStruct IndData;

/*
 * structure for environment
 * -------------------------
 */

struct EnvStruct {
  IndData individuals[MAX_IND];
  float   *landscape;
  float   *staticLandscape;
  int     xSize;
  int     ySize;
  int     nInd;
  int     nHills;
  float   sigma;
  int     updateVelocity;
  float   noise;
  int     latency;
  int     output;
  int     shape;
  int     startSequence;
  int     running;
  int     started;
  float   alpha;

  char  filename[256];
  int   landscapeNr;
  int   nSequence;
  int   frameSkip;
  int   frameSkipCounter;
};

typedef struct EnvStruct EnvData;

/*
 * structure for different choices
 * -------------------------------
 */

struct ChoiceStruct {
  char *text;
  void *f;
};

typedef struct ChoiceStruct ChoiceData;

/*
 * structure for menu
 * ------------------
 */

struct MenuStruct {
  char  *text;
  int   nInd;
  int   updateVelocity;
  int   latency;
  int   shape;
  int   nHills;
};

typedef struct MenuStruct MenuData;

/*
 * global variables
 * ----------------
 */

#define INPUT_INT   1
#define INPUT_FLOAT 2

#define WINDOW_WIDTH  700
#define WINDOW_HEIGHT 700
#define GWINDOW_WIDTH  368
#define GWINDOW_HEIGHT 368

  GLint   grabX=0;
  GLint   grabY=0;
  GLint   grabWidth=WINDOW_WIDTH;
  GLint   grabHeight=WINDOW_HEIGHT-96;

  GLfloat angle1=0.0;
  GLfloat angle2=30.0;
  GLint   cursorX;
  GLint   cursorY;
  GLint   cursorXorigin=7;
  GLint   cursorYorigin=0;
  int     inputMode=0;
  char    inputKey=' ';
  int     inputType=0;
  void    *inputPntr=NULL;
  void    (*inputFunc)(void)=NULL;
  int     inputPos=0;
  char    inputBuffer[256];
  int     displayMenu=1;
  int     displayInd=2;
  int     displayLines=0;
  GLint   windowWidth=WINDOW_WIDTH;
  GLint   windowHeight=WINDOW_HEIGHT;

  int   maxSelection=9;
  MenuData menu[32]={
  {"              Species  Update rule  Deformation  Latency  Hills",
   0,0,0,0,0},
  {"Experiment 1:    40      Constant    Gaussian       0       0",
   40,0, 0,0, 0},
  {"Experiment 2:    40      Constant    Gaussian      50       0",
   40,0,50,0, 0},
  {"Experiment 3:    40    Proportional  Gaussian       0       0",
   40,1, 0,0, 0},
  {"Experiment 4:    40    Proportional  Gaussian      50       0",
   40,1,50,0, 0},
  {"Experiment 5:    40     Integrated   Gaussian       0       0",
   40,2, 0,0, 0},
  {"Experiment 6:    40     Integrated   Gaussian       2       0",
   40,2, 2,0, 0},
  {"Experiment 7:    40     Integrated   Gaussian      50       0",
   40,2,50,0, 0},
  {"Experiment 8:    40     Integrated   Gaussian       0     100",
   40,2, 0,0,100},
  {"Experiment 9:    40     Integrated   Gaussian      50     100",
   40,2,50,0,100}
  };
  EnvData *env;

int defW=5;
int defH=5;
float *hat[25];
float *dgaussian[25];
float *gaussian[25];

void
velocityConstant(float *land,int xSize,int ySize,
                 float x,float y,float vx,float vy,float *newVx,float *newVy);
void
velocityProportional(float *land,int xSize,int ySize,
                 float x,float y,float vx,float vy,float *newVx,float *newVy);
void
velocityIntegratedDamped(float *land,int xSize,int ySize,
                 float x,float y,float vx,float vy,float *newVx,float *newVy);
void
velocityIntegratedUndamped(float *land,int xSize,int ySize,
                 float x,float y,float vx,float vy,float *newVx,float *newVy);

ChoiceData shapeChoice[MAX_NSHAPES]={
    {"Gaussian", NULL},
    {"Hat",      NULL},
    {"DGaussian",NULL}
};
ChoiceData updateVelocityChoice[MAX_UDV]={
    {"Constant",         (void *)NULL},
    {"Proportional",     (void *)NULL},
    {"Integrated Damped",(void *)NULL}
};
ChoiceData displayIndChoice[MAX_NINDDIS]={
    {"No Individuals",(void *)NULL},
    {"Points",        (void *)NULL},
    {"Spheres",       (void *)NULL}
};

/*
 * get random integer with range [l,r]
 * -----------------------------------
 * parameters
 *   int l: lower bound
 *   int r: upper bound
 * returns
 *   int    random integer with range [l,r]
 */

int
getRandomInt(int l,int r)
{
  return (int) ((r-l+1.0)*drand48())+l;
}

/*
 * calculate gradient of landscape
 * -------------------------------
 * parameters
 *   float *land: pointer to landscape
 *   int   xSize: size of landscape in x direction
 *   int   ySize: size of landscape in y direction
 *   int   x:     x position on landscape
 *   int   y:     y position on landscape
 *   float *gx:   gradient in x direction
 *   float *gy:   gradient in y direction
 * returns
 *   void:         nothing
 */

void
gradient(float *land,int xSize,int ySize,int x,int y,float *gx,float *gy)
{
  int x0;
  int x1;
  int x2;
  int y0;
  int y1;
  int y2;

  x0=x-1;
  x1=x;
  x2=x+1;
  if (x0<0)
    x0+=xSize;
  if (x2>=xSize)
    x2-=xSize;
  y0=y-1;
  y1=y;
  y2=y+1;
  if (y0<0)
    y0+=ySize;
  if (y2>=ySize)
    y2-=ySize;
  *gx=(land[y*xSize+x2]-land[y*xSize+x1]+
       land[y*xSize+x1]-land[y*xSize+x0])/2.0;
  *gy=(land[y2*xSize+x]-land[y1*xSize+x]+
       land[y1*xSize+x]-land[y0*xSize+x])/2.0;
}

/*
 * velocity update method: constant velocity
 * -----------------------------------------
 * parameters
 *   float *land:  pointer to landscape
 *   int   xSize:  size of landscape in x direction
 *   int   ySize:  size of landscape in y direction
 *   float x:      position on landscape
 *   float y:      position on landscape
 *   float vx:     velocity of individual
 *   float vy:     velocity of individual
 *   float *newVx: velocity of individual
 *   float *newVy: velocity of individual
 * returns
 *   void:         nothing
 */

void
velocityConstant(float *land,int xSize,int ySize,
                 float x,float y,float vx,float vy,float *newVx,float *newVy)
{
  float gx;
  float gy;
  float alpha=1.0;

  gradient(land,xSize,ySize,x,y,&gx,&gy);

  if (gx<0.0)
    *newVx=-alpha;
  else if (gx>0.0)
    *newVx=alpha;
  else
    *newVx=0.0;

  if (gy<0.0)
    *newVy=-alpha;
  else if (gy>0.0)
    *newVy=alpha;
  else
    *newVy=0.0;
}

/*
 * velocity update method: velocity is proportional to the landscape's gradient
 * ----------------------------------------------------------------------------
 * parameters
 *   float *land:  pointer to landscape
 *   int   xSize:  size of landscape in x direction
 *   int   ySize:  size of landscape in y direction
 *   float x:      position on landscape
 *   float y:      position on landscape
 *   float vx:     velocity of individual
 *   float vy:     velocity of individual
 *   float *newVx: velocity of individual
 *   float *newVy: velocity of individual
 * returns
 *   void:         nothing
 */

void
velocityProportional(float *land,int xSize,int ySize,
                     float x,float y,float vx,float vy,
                     float *newVx,float *newVy)

{
  float alpha=10.0;
  float gx;
  float gy;

  gradient(land,xSize,ySize,x,y,&gx,&gy);
  *newVx=alpha*gx;
  *newVy=alpha*gy;
}

/*
 * velocity update method: integrated and damped
 * ---------------------------------------------
 * parameters
 *   float *land:  pointer to landscape
 *   int   xSize:  size of landscape in x direction
 *   int   ySize:  size of landscape in y direction
 *   float x:      position on landscape
 *   float y:      position on landscape
 *   float vx:     velocity of individual
 *   float vy:     velocity of individual
 *   float *newVx: velocity of individual
 *   float *newVy: velocity of individual
 * returns
 *   void:         nothing
 */

void
velocityIntegratedDamped(float *land,int xSize,int ySize,
                         float x,float y,float vx,float vy,
                         float *newVx,float *newVy)
{
  float alpha=10.0;
  float gx;
  float gy;

  gradient(land,xSize,ySize,x,y,&gx,&gy);
  *newVx=vx+alpha*gx;
  *newVy=vy+alpha*gy;
  *newVx-=0.1*(*newVx);
  *newVy-=0.1*(*newVy);
}

/*
 * velocity update method: integrated
 * ----------------------------------
 * parameters
 *   float *land:  pointer to landscape
 *   int   xSize:  size of landscape in x direction
 *   int   ySize:  size of landscape in y direction
 *   float x:      position on landscape
 *   float y:      position on landscape
 *   float vx:     velocity of individual
 *   float vy:     velocity of individual
 *   float *newVx: velocity of individual
 *   float *newVy: velocity of individual
 * returns
 *   void:         nothing
 */

void
velocityIntegratedUndamped(float *land,int xSize,int ySize,
                           float x,float y,float vx,float vy,
                           float *newVx,float *newVy)
{
  float alpha=1.0;
  float gx;
  float gy;

  gradient(land,xSize,ySize,x,y,&gx,&gy);
  *newVx=vx+alpha*gx;
  *newVy=vy+alpha*gy;
}

/*
 * individuals are moved according to their velocity
 * -------------------------------------------------
 * parameters
 *   float *land:          pointer to landscape
 *   int xSize:            size in x direction
 *   int ySize:            size in y direction
 *   IndData *individuals: pointer to individuals
 *   int nInd:             number of individuals
 * returns
 *   void:                 nothing
 */

void
moveIndividuals(float *land,int xSize,int ySize,IndData *individuals,int nInd)
{
  int i;
  float x;
  float y;
  IndData *ind;

  for (i=0;i<nInd;i++) {
    ind=&individuals[i];
    x=ind->x;
    y=ind->y;
    ind->xOld[ind->i]=x;
    ind->yOld[ind->i]=y;
    ind->i++;
    if (ind->i>=MAX_HIST)
      ind->i=0;
    x+=0.1*(ind->vx+ind->noise*(drand48()-0.5));
    y+=0.1*(ind->vy+ind->noise*(drand48()-0.5));
    while (x>=xSize)
      x-=xSize;
    while (x<0)
      x+=xSize;
    while (y>=ySize)
      y-=ySize;
    while (y<0)
      y+=ySize;
    ind->x=x;
    ind->y=y;
    switch (ind->updateVelocity) {
    case 0:
      velocityConstant(land,xSize,ySize,x,y,ind->vx,ind->vy,
                       &ind->vx,&ind->vy);
      break;
    case 1:
      velocityProportional(land,xSize,ySize,x,y,ind->vx,ind->vy,
                           &ind->vx,&ind->vy);
      break;
    case 2:
      velocityIntegratedDamped(land,xSize,ySize,x,y,ind->vx,ind->vy,
                               &ind->vx,&ind->vy);
      break;
    case 3:
      velocityIntegratedUndamped(land,xSize,ySize,x,y,ind->vx,ind->vy,
                                 &ind->vx,&ind->vy);
      break;
    }
  }
}

/*
 * copy landscape
 * --------------
 * parameters
 *   float *dstLand:  pointer to landscape
 *   int xSize:       size of landscape in x direction
 *   int ySize:       size of landscape in y direction
 *   float *srcLand:  pointer to source landscape
 * returns
 *   void:            nothing
 */

void
copyLandscape(float *dstLand,int xSize,int ySize,float *srcLand)
{
  int size;
  int i;

  size=xSize*ySize;
  for (i=0;i<size;i++)
    *dstLand++=*srcLand++;
}

/*
 * handle interaction between individuals and environment
 * ------------------------------------------------------
 * parameters
 *   float *land:       resulting landscape
 *   int xSize:         size in x direction
 *   int ySize:         size in y direction
 *   float *staticLand: static landscape
 *   IndData *ind:      pointer to individuals
 *   int nInd:          number of individuals
 * returns
 *   void:              nothing
 */

void
envInteraction(float *land,int xSize,int ySize,float *staticLand,
               IndData *individuals,int nInd)
{
  IndData *ind;
  int x;
  int y;
  int i;
  int dx;
  int dy;
  int index;
  int deformXhalf=xSize/2;
  int deformYhalf=ySize/2;
  float *pSrc;
  float *pDst;
  int   size;
  float fx;
  float fy;
  float dlx;
  float dly;
  int l;

  copyLandscape(land,xSize,ySize,staticLand);
  size=xSize*ySize;
  for (i=0;i<nInd;i++) {
    ind=&individuals[i];
    index=ind->i-ind->l;
    if (index<0)
      index+=MAX_HIST;
    fx=ind->xOld[index]+(1.0/(2*defW));
    fy=ind->yOld[index]+(1.0/(2*defH));
    x=(int)fx;
    y=(int)fy;
    dlx=fx-x;
    dly=fy-y;
    l=(int)(dly*defH);
    l*=defW;
    l+=(int)(dlx*defW);
    assert(l>=0 && l<defW*defH);
    x-=deformXhalf;
    if (x<0)
      x+=xSize;
    y-=deformYhalf;
    if (y<0)
      y+=xSize;

    pDst=&land[y*xSize+x];
    pSrc=ind->deform[l];
    for (dy=0;dy<ySize;dy++) {
      for (dx=0;dx<xSize;dx++) {
        *pDst++-=*pSrc++;
        x++;
        if (x>=xSize) {
          x-=xSize;
          pDst-=xSize;
	}
      }
      y++;
      pDst+=xSize;
      if (y>=ySize) {
        y-=ySize;
        pDst-=size;
      }
    }
  }
}

/*
 * place individuals on landscape (after they have moved)
 * ------------------------------------------------------
 * parameters
 *   float *land:  pointer to landscape
 *   int xSize:    size of landscape in x direction
 *   int ySize:    size of landscape in y direction
 *   IndData *ind: pointer to individuals
 *   int nInd:     number of individuals
 * returns
 *   void:         nothing
 */

void
placeIndividualsOnLandscape(float *land,int xSize,int ySize,
                            IndData *ind,int nInd)
{
  int i;
  int x;
  int y;
  float lx;
  float ly;
  float gx=0.0;
  float gy=0.0;

  for (i=0;i<nInd;i++) {
    lx=ind[i].x;
    ly=ind[i].y;
    x=lx;
    y=ly;
    if (x>=xSize)
      x-=xSize;
    if (y>=ySize)
      y-=ySize;
    gradient(land,xSize,ySize,x,y,&gx,&gy);
    ind[i].z=land[y*xSize+x]+(lx-x)*gx+(ly-y)*gy;
  }
}

/*
 * read in landscape from file
 * ---------------------------
 * parameters
 *   char *filename: pointer to filename
 *   float *land:    pointer to landscape
 *   int xSize:      size in x direction
 *   int ySize:      size in y direction
 * returns
 *   void:           nothing
 */

void
readLandFile(char *filename,float *land,int xSize,int ySize)
{
  FILE *fd;
  int x;
  int y;
  int i;

  fd=fopen(filename,"r");
  if (fd==NULL) {
    fprintf(stderr,"Could not read landscape file %s\n",filename);
    exit(0);
  }
  i=0;
  for (y=0;y<ySize;y++) {
    for (x=0;x<xSize;x++) {
      fscanf(fd,"%f\n",&(land[i++]));
    }
    fscanf(fd,"\n");
  }
  fclose(fd);
}

/*
 * write landscape to file
 * -----------------------
 * parameters
 *   char *filename: pointer to filename
 *   float *land:    pointer to landscape
 *   int xSize:      size in x direction
 *   int ySize:      size in x direction
 * returns
 *   void:           nothing
 */

void
writeLandFile(char *filename,float *land,int xSize,int ySize)
{
  FILE *fd;
  int x;
  int y;
  int i;
  fd=fopen(filename,"w");
  if (fd==NULL) {
    fprintf(stderr,"Could not read landscape file %s\n",filename);
    exit(0);
  }
  i=0;
  for (y=0;y<ySize;y++) {
    for (x=0;x<xSize;x++) {
      fprintf(fd,"%g\n",land[i++]);
    }
    fprintf(fd,"\n");
  }
  fclose(fd);
}

/*
 * write positions of individuals to file
 * --------------------------------------
 * parameters
 *   char *filename: pointer to filename
 *   IndData *ind:   pointer to individuals
 *   int nInd:       number of individuals
 * returns
 *   void:           nothing
 */

void
writeIndFile(char *filename,IndData *ind,int nInd)
{
  FILE *fd;
  int i;

  fd=fopen(filename,"w");
  if (fd==NULL) {
    fprintf(stderr,"Could not write individual %s\n",filename);
    exit(0);
  }
  for (i=0;i<nInd;i++) {
    fprintf(fd,"%f %f %f\n",ind[i].x,ind[i].y,ind[i].z);
  }
  fclose(fd);
}

/*
 * init landscape
 * --------------
 * paramerters
 *   float *land:     pointer to landscape
 *   int xSize:       size of landscape in x direction
 *   int ySize:       size of landscape in y direction
 *   int nHills:      number of hills
 *   float *deform:   pointer to deformation
 * returns
 *   void:            nothing
 */

void
initLandscape(float *land,int xSize,int ySize,int nHills,float **deform)
{
  int i;
  int dx;
  int dy;
  int x;
  int y;
  int deformXhalf=xSize/2;
  int deformYhalf=ySize/2;
  float *pSrc;
  float *pDst;
  int   size;

  pDst=land;
  size=xSize*ySize;
  for (i=0;i<size;i++) {
    *pDst++=0;
  }

  for (i=0;i<nHills;i++) {
    x=getRandomInt(0,xSize-1);
    y=getRandomInt(0,ySize-1);
    x-=deformXhalf;
    if (x<0)
      x+=xSize;
    y-=deformYhalf;
    if (y<0)
      y+=ySize;

    pDst=&land[y*xSize+x];
    /*
    pSrc=&(deform[getRandomInt(0,defW*defH-1)][0]);
    */
    pSrc=&(deform[0][0]);
    for (dy=0;dy<ySize;dy++) {
      for (dx=0;dx<xSize;dx++) {
        *pDst++ += *pSrc++;
        x++;
        if (x>=xSize) {
          x-=xSize;
          pDst-=xSize;
	}
      }
      y++;
      pDst+=xSize;
      if (y>=ySize) {
        y-=ySize;
        pDst-=size;
      }
    }
  }
}

/*
 * init one individual
 * -------------------
 * parameters
 *   IndData *ind:  pointer to individuals
 *   float x:       x coordinate of individual
 *   float y:       y coordinate of individual
 *   float z:       z coordinate of individual
 *   int updateVelocity: update rule for velocity
 *   float noise:   noise to apply after movement
 *   float l:       latency
 *   float *deform: shape of individual
 * returns
 *   void:          nothing
 */

void
initOneIndividual(IndData *ind,float x,float y,float z,int updateVelocity,
                  float noise,float l,float **deform)
{
  int i;

  ind->x=x;
  ind->y=y;
  ind->z=z;
  ind->vx=0;
  ind->vy=0;
  ind->updateVelocity=updateVelocity;
  ind->l=l;
  ind->noise=noise;
  ind->i=0;
  for (i=0;i<MAX_HIST;i++) {
    ind->xOld[i]=x;
    ind->yOld[i]=y;
  }
  ind->deform=deform;
}

/*
 * init all individuals
 * --------------------
 * parameters
 *   IndData *ind:   pointer to individuals
 *   int nInd:       number of individuals
 *   int updateVelocity: update rule for velocity
 *   float noise:    noise to apply after movement
 *   float l:        latency
 *   float *land:    pointer to landscape
 *   int xSize:      size of landscape in x direction
 *   int ySize:      size of landscape in y direction
 *   float *deform:  shape of individuals
 * returns
 *   void:           nothing
 */

void
initIndividuals(IndData *ind,int nInd,int updateVelocity,
                float noise,float l,float *land,int xSize,int ySize,
                int deformType)
{
  int i;
  float x;
  float y;
  float **deform;

  switch (deformType) {
    case 0: deform=gaussian;break;
    case 1: deform=dgaussian;break;
    case 2: deform=hat;break;
    default: printf("Error.\n");exit(0);break;
  }

  for (i=0;i<nInd;i++) {
    x=getRandomInt(0,xSize-1);
    y=getRandomInt(0,ySize-1);
    initOneIndividual(&ind[i],x,y,land[(int)(y*xSize+x)],
                      updateVelocity,noise,l,deform);
  }
}

/*
 * save postscript landscape
 * -------------------------
 * parameters
 *   EnvData *env:   pointer to environment
 *   char *filename: pointer to filename
 * returns
 *   void:         nothing
 */

void
savePPMLandscape(EnvData *env,char *filename)
{
  FILE *fd;
  unsigned char buffer1[1600*1600*3];
  unsigned char buffer2[1600*1600*3];
  int x;
  int y;
  int lineSize;
  int imageSize;
  unsigned char *p1;
  unsigned char *p2;

  glReadPixels(grabX,grabY,
               grabWidth,grabHeight,GL_RGB,GL_UNSIGNED_BYTE,buffer1);

  imageSize=grabWidth*grabHeight*3;
  lineSize=grabWidth*3;
  p1=buffer1;
  p2=buffer2+imageSize-lineSize;
  for (y=0;y<grabHeight;y++) {
    for (x=0;x<lineSize;x++) {
      *p2++=*p1++;
    }
    p2-=lineSize<<1;
  }

  fd=fopen(filename,"w");
  if (fd==NULL)
    fprintf(stderr,"Could not write file %s\n",filename);
  else {
    fprintf(fd,"P6\n");
    fprintf(fd,"#2D Deformable Landscape Version 1.0 by Marc Ebner\n");
    fprintf(fd,"# Individuals:                  %d\n",env->nInd);
    fprintf(fd,"# Hills:                        %d\n",env->nHills);
    fprintf(fd,"# Size of Landscape (X):        %d\n",env->xSize);
    fprintf(fd,"# Size of Landscape (Y):        %d\n",env->ySize);
    fprintf(fd,"# Width of Gaussians:           %f\n",env->sigma);
    fprintf(fd,"# Velocity Update:              %s\n",
                 updateVelocityChoice[env->updateVelocity].text);
    fprintf(fd,"# Noise:                        %f\n",env->noise);
    fprintf(fd,"# Latency:                      %d\n",env->latency);
    fprintf(fd,"# Frame Skip:                   %d\n",env->frameSkip);

    fprintf(fd,"%d %d\n",grabWidth,grabHeight);
    fprintf(fd,"%d\n",255);
    if (grabWidth*grabHeight!=(int)
        fwrite(buffer2,3*sizeof(unsigned char),grabWidth*grabHeight,fd)) {
      fprintf(stderr,"Error during writing of file %s\n",filename);      
      fclose(fd);
      return;
    }
    if (fclose(fd)==EOF) {
      fprintf(stderr,"Could not close file %s\n",filename);
      return;
    }
  }
}

/*
 * save results
 * ------------
 * parameters
 *   EnvData *env: pointer to environment
 * returns
 *   void:         nothing
 */

void
saveResults(EnvData *env)
{
  char filename[256];

  sprintf(filename,"%slandscape%04d.ppm",env->filename,env->landscapeNr);
  env->landscapeNr++;
  savePPMLandscape(env,filename);
}

/*
 * print string in window
 * ----------------------
 * parameters
 *  char *format: format of string
 *  ...           remaining arguments
 * returns
 *   void:        nothing
 */

void
glPrintf(char *format,...)
{
  va_list args;
  char string[256];
  char *s;

  va_start(args,format);
  vsprintf(string,format,args);
  for (s=string;*s;s++) {
    if (*s=='\n') {
      cursorY-=3;
      cursorX=cursorXorigin;
      glRasterPos2i(cursorX,cursorY);
    } else
      glutBitmapCharacter(GLUT_BITMAP_9_BY_15,*s);
  }
}

/*
 * show menu on screen
 * -------------------
 * parameters
 *   MenuData *menu:   pointer to menu data
 *   int maxSelection: number of predefined settings
 *   EnvData *env:     pointer to environment data
 * returns
 *   void:             nothing
 */

void
showMenu(MenuData *menu,int maxSelection,EnvData *env)
{
  GLfloat textEmissive[]={1.0,1.0,1.0,1.0};
  int   i;

  /* free letters:
   * ABCDEFGHIJKLMNOPQRSTUVWXYZ
   * AB       J            
   */

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluOrtho2D(0.0,131.0,-131.0,0.0);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,textEmissive);
  glColor3f(1.0,1.0,1.0);
  cursorX=cursorXorigin;
  cursorY=cursorYorigin;
  glRasterPos2i(cursorX,cursorY);

  glPrintf("\n");
  glPrintf("                  2D Deformable Landscape Program\n");
  glPrintf("                  copyright (C) 2000 by Marc Ebner\n");
  glPrintf("        Universitaet Wuerzburg, Lehrstuhl fuer Informatik II\n");
  glPrintf("                Am Hubland, 97074 Wuerzburg, Germany\n");
  glPrintf("   ebner@informatik.uni-wuerzburg.de or m.ebner.1@alumni.nyu.edu\n");
  glPrintf("-------------------------------------------------------------------\n");
  glPrintf("I) Species:                      %d\n",env->nInd);
  glPrintf("H) Number of Hills               %d\n",env->nHills);
  glPrintf("D) Shape of Deformation:         %s\n",
               shapeChoice[env->shape].text);
  glPrintf("X) Size of Landscape in X Dir.   %d\n",env->xSize);
  glPrintf("Y) Size of Landscape in Y Dir.   %d\n",env->ySize);
  glPrintf("G) Width of Gaussians:           %f\n",env->sigma);
  glPrintf("V) Velocity Update:              %s\n",
               updateVelocityChoice[env->updateVelocity].text);
  glPrintf("N) Noise:                        %f\n",env->noise);
  glPrintf("L) Latency:                      %d\n",env->latency);
  glPrintf("R) Run                            Left Arrow) Rotate Left\n");
  glPrintf("S) Stop                          Right Arrow) Rotate Right\n");
  glPrintf("C) Continue                         Up Arrow) Rotate Up\n");
  glPrintf("E) End Run                        Down Arrow) Rotate Down\n");
  glPrintf("Z) Run and Save Sequence\n");
  glPrintf("P) Save Screen with Filename %slandscape%04d.ppm\n",
         env->filename,env->landscapeNr);
  glPrintf("K) Number of Frames in Sequence: %d\n",env->nSequence);
  glPrintf("W) Save Sequence of %d Landscapes\n",env->nSequence);
  glPrintf("F) Frame Skip:                   %d\n",env->frameSkip);
  glPrintf("O) Toggle Output:                %s\n",
           (env->output)? "Output On":"Output Off");
  glPrintf("M) Toggle Menu:                  %s\n",
           (displayMenu)? "Menu On":"Menu Off");
  glPrintf("T) Display Individuals as:       %s\n",
           displayIndChoice[displayInd].text);
  glPrintf("U) Draw Landscape:               %s\n",
           (displayLines)? "Lines":"Solid");
  glPrintf("Q) Exit\n");
  glPrintf("+) Save Landscape\n");
  glPrintf("-) Load Landscape\n");

  glPrintf("    %s\n",menu[0].text);
  for (i=1;i<=maxSelection;i++) {
    glPrintf("%d.) %s\n",i,menu[i].text);
  }

  if (inputMode)
    glPrintf("\nInput %c) %s_\n",inputKey,inputBuffer);

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(55.0,1.0*(GLfloat)windowWidth/(GLfloat)windowHeight,
                 1.0,1000.0);
  glMatrixMode(GL_MODELVIEW);
}

/*
 * get integer value from user
 * ---------------------------
 * parameters
 *   char key:        menu selection
 *   int *value:      pointer to place where value will be stored
 *   void (*f)(void): pointer to funtion which has to be executed when done
 * returns
 *   void:       nothing
 */

void
getInputInt(char key,int *value,void (*f)(void))
{
  inputMode=1;
  inputKey=key;
  inputPos=0;
  inputBuffer[0]=0;
  inputPntr=value;
  inputType=INPUT_INT;
  inputFunc=f;
  displayMenu=1;
  glutPostRedisplay();
}

/*
 * get float value from user
 * -------------------------
 * parameters
 *   char key:        menu selection
 *   float *value:    pointer to place where value will be stored
 *   void (*f)(void): pointer to funtion which has to be executed when done
 * returns
 *   void:         nothing
 */

void
getInputFloat(char key,float *value,void (*f)(void))
{
  inputMode=1;
  inputKey=key;
  inputPos=0;
  inputBuffer[0]=0;
  inputPntr=value;
  inputType=INPUT_FLOAT;
  inputFunc=f;
  displayMenu=1;
  glutPostRedisplay();
}

/*
 * get next selection (from a list of choices) from user
 * -----------------------------------------------------
 * parameters
 *   int y:      y coordinate of cursor
 *   int x:      x coordinate of cursor
 *   int *value: pointer to place where selection will be stored
 * returns
 *   void:       nothing
 */

void
getNextSelection(int *value,int n)
{
  (*value)++;
  if (*value==n)
    *value=0;
  glutPostRedisplay();
}

/*
 * init positions of individuals
 * -----------------------------
 * parameters
 *   EnvData *env: pointer to environment
 * returns
 *   void:        nothing
 */

void
initIndPos(EnvData *env)
{
  int j;

  for (j=0;j<env->nInd;j++) {
    env->individuals[j].x=0.0;
    env->individuals[j].y=0.0;
    env->individuals[j].z=0.0;
  }
}

/*
 * init deformations
 * -----------------
 * parameters
 *   float sigma: width of gaussian
 * returns
 *   void:        nothing
 */

void
initShapes(void)
{
  int i;
  int j;
  int k;
  int l;
  float dlx;
  float dly;
  float x;
  float y;
  float sigma;
  int deformXhalf=env->xSize/2;
  int deformYhalf=env->ySize/2;

  sigma=env->sigma;

  l=0;
  for (dly=0;dly<1.0;dly+=1.0/defH) {
   for (dlx=0;dlx<1.0;dlx+=1.0/defW) {
    if (gaussian[l]!=NULL)
      free(gaussian[l]);
    gaussian[l]=(float *)malloc(env->xSize*env->ySize*sizeof(float));
    assert(gaussian[l]!=NULL);
    k=0;
    for (j=0;j<env->ySize;j++) {
      for (i=0;i<env->xSize;i++) {
        x=i-deformXhalf;
        y=j-deformYhalf;
        x-=dlx;
        y-=dly;
        gaussian[l][k++]=exp(-sigma*(x*x+y*y));
      }
    }

    if (dgaussian[l]!=NULL)
      free(dgaussian[l]);
    dgaussian[l]=(float *)malloc(env->xSize*env->ySize*sizeof(float));
    assert(dgaussian[l]!=NULL);
    k=0;
    for (j=0;j<env->ySize;j++) {
      for (i=0;i<env->xSize;i++) {
        x=i-deformXhalf;
        y=j-deformYhalf;
        x-=dlx;
        y-=dly;
        dgaussian[l][k++]=2*sqrt(sigma)*(x+y)*exp(-sigma*(x*x+y*y));
      }
    }

    if (hat[l]!=NULL)
      free(hat[l]);
    hat[l]=(float *)malloc(env->xSize*env->ySize*sizeof(float));
    assert(hat[l]!=NULL);
    k=0;
    for (j=0;j<env->ySize;j++) {
      for (i=0;i<env->xSize;i++) {
        x=i-deformXhalf;
        y=j-deformYhalf;
        x-=dlx;
        y-=dly;
        hat[l][k++]=exp(-sigma*(x*x+y*y))-1.5*exp(-sigma*5*(x*x+y*y));
      }
    }
    l++;
   }
  }
}

/*
 * create environment
 * ------------------
 * parameters
 *   int xSize:   size of landscape in x direction
 *   int ySize:   size of landscape in y direction
 *   int nHills:  number of hills
 *   float sigma: width of gaussians
 * returns
 *   void:        nothing
 */

void
createEnvironment(int xSize,int ySize,int nHills,float sigma)
{
  int i;

  env=(EnvData *)malloc(sizeof(EnvData));
  assert(env!=NULL);
  env->nInd=30;
  env->nHills=nHills;
  env->xSize=xSize;
  env->ySize=ySize;
  env->sigma=sigma;
  env->latency=50;
  env->updateVelocity=2;
  env->output=1;
  env->startSequence=0;
  env->running=0;
  env->started=0;

  env->landscape=(float *)malloc(xSize*ySize*sizeof(float));
  env->staticLandscape=(float *)malloc(xSize*ySize*sizeof(float));

  for (i=0;i<defW*defH;i++) {
    gaussian[i]=NULL;
    dgaussian[i]=NULL;
    hat[i]=NULL;
  }

  initShapes();
  initLandscape(env->staticLandscape,xSize,ySize,nHills,gaussian);
  initIndPos(env);
  copyLandscape(env->landscape,env->xSize,env->ySize,env->staticLandscape);

  sprintf(env->filename,"land/");
  env->landscapeNr=0;
  env->nSequence=51;
  env->frameSkip=1;
  env->frameSkipCounter=env->frameSkip;
}

/*
 * destroy environment
 * -------------------
 * parameters
 *   EnvData *env: pointer to environment
 * returns
 *   void:         nothing
 */

void
destroyEnvironment(EnvData *env)
{
  free(env);
}

/*
 * set parameters of environment accordig to settings in menu
 * ----------------------------------------------------------
 * parameters
 *   EnvData *env:   pointer to environment
 *   MenuData *menu: pointer to menu data
 *   int selection:  selection among predefined settings
 * returns
 *   void:           nothing
 */

void
setParameters(EnvData *env,MenuData *menu,int selection)
{
  env->nInd=menu[selection].nInd;
  env->updateVelocity=menu[selection].updateVelocity;
  env->latency=menu[selection].latency;
  env->shape=menu[selection].shape;
  env->nHills=menu[selection].nHills;
}

/*
 * display landscape
 * -----------------
 * parameters
 *   void: nothing
 * returns
 *   void: nothing
 */

void
display(void)
{
#ifdef USEGRID
  static int init=1;
  static GLfloat grid[LANDSCAPE_YSIZE][LANDSCAPE_XSIZE][3];
#endif

  GLfloat landAmbient[]={0.67,0.45,0.32,1.0};
  GLfloat landDiffuse[]={0.67,0.45,0.32,1.0};
  GLfloat landSpecular[]={1.0,1.0,1.0,1.0};
  GLfloat landEmissive[]={0.0,0.0,0.0,1.0};
  GLfloat landShininess=1.0*128;

  GLfloat indAmbient[]={0.0,0.5,0.0,1.0};
  GLfloat indDiffuse[]={0.0,0.5,0.0,1.0};
  GLfloat indSpecular[]={1.0,1.0,1.0,1.0};
  GLfloat indEmissive[]={0.0,0.3,0.0,1.0};
  GLfloat indShininess=1.0*128;

  GLfloat lightPosition[]={0.0,0.0,1.0,0.0};

  int x;
  int y;
  int xSize;
  int ySize;
  float *land;
  float *landXY;
  float *landXp1Y;
  float *landXYp1;
  float *landXYp2;
  float *landXp1Yp1;
  IndData *ind;
  int i;
  int size;

  xSize=env->xSize;
  ySize=env->ySize;

  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

  if (displayMenu)
    showMenu(menu,maxSelection,env);

  glLoadIdentity();

  gluLookAt(0.0,0.0,7.0,0.0,0.0,0.0,0.0,1.0,0.0);
  glRotatef(angle1,0.0,cos(2*M_PI*angle2/360.0),sin(2*M_PI*angle2/360.0));
  glRotatef(angle2,1.0,0.0,0.0);

  glRotatef(-90.0,1.0,0.0,0.0);
  glScalef(5.0/xSize,5.0/ySize,0.2);
  glLightfv(GL_LIGHT0,GL_POSITION,lightPosition);
  glTranslatef(-xSize/2.0,-ySize/2.0,0.0);

  glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT,landAmbient);
  glMaterialfv(GL_FRONT_AND_BACK,GL_DIFFUSE,landDiffuse);
  glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,landSpecular);
  glMaterialf(GL_FRONT_AND_BACK,GL_SHININESS,landShininess);
  glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,landEmissive);

  land=env->landscape;
#ifdef USEGRID
  glColor3f(1.0,1.0,1.0);
  glMaterialfv(GL_FRONT_AND_BACK,GL_EMISSION,indEmissive);
  if (init) {
    init=0;
    for (y=0;y<ySize;y++) {
      for (x=0;x<xSize;x++) {
        grid[y][x][0]=x;
        grid[y][x][1]=y;
        grid[y][x][2]=0;
      }
    }
    glMap2f(GL_MAP2_VERTEX_3,0,1,3,xSize,
                             0,1,LANDSCAPE_XSIZE*3,ySize,&grid[0][0][0]);
    glEnable(GL_MAP2_VERTEX_3);
    glEnable(GL_AUTO_NORMAL);
    glMapGrid2f(100,0,1,
                100,0,1);
  }
  for (y=0;y<ySize-1;y++) {
    for (x=0;x<xSize;x++) {
      grid[y][x][2]=land[y*xSize+x];
    }
  }
  glEvalMesh2(GL_LINE,0,100,0,100);
#else
  landXY=land;
  size=xSize*ySize;
  for (y=0;y<ySize-1;y++) {
    landXp1Y=landXY+1;
    landXYp1=landXY+xSize;
    if (y==xSize-2)
      landXYp2=land;
    else
      landXYp2=landXYp1+xSize;
    landXp1Yp1=landXYp1+1;
    if (displayLines)
      glBegin(GL_LINE_STRIP);
    else
      glBegin(GL_TRIANGLE_STRIP);
    for (x=0;x<xSize;x++) {
      if (x==xSize-1)
        landXp1Y-=xSize;
      glNormal3f(*landXp1Yp1-*landXYp1,
                 *landXYp2-*landXYp1,1.0);
      glVertex3f((GLfloat)x,(GLfloat)(y+1),*landXYp1);

      glNormal3f(*landXp1Y-*landXY,
                 *landXYp1-*landXY,1.0);
      glVertex3f((GLfloat)x,(GLfloat)y,*landXY);
      landXY++;
      landXp1Y++;
      landXYp1++;
      landXYp2++;
      landXp1Yp1++;
    }
    glEnd();
  }
#endif

  if (displayInd==1) {
    glMaterialfv(GL_FRONT,GL_AMBIENT,indAmbient);
    glMaterialfv(GL_FRONT,GL_DIFFUSE,indDiffuse);
    glMaterialfv(GL_FRONT,GL_SPECULAR,indSpecular);
    glMaterialf(GL_FRONT,GL_SHININESS,indShininess);
    glMaterialfv(GL_FRONT,GL_EMISSION,indEmissive);
    ind=env->individuals;
    /*
    glDepthFunc(GL_ALWAYS);
    */
    for (i=0;i<env->nInd;i++) {
      glBegin(GL_TRIANGLE_FAN);
        glNormal3f(0.0,0.0,1.0);
        glVertex3f(ind[i].x,ind[i].y,ind[i].z+0.3);
        glNormal3f(1.0,0.0,1.0);
        glVertex3f(ind[i].x+1.0,ind[i].y,ind[i].z);
        glNormal3f(0.0,1.0,1.0);
        glVertex3f(ind[i].x,ind[i].y+1.0,ind[i].z);
        glNormal3f(-1.0,0.0,1.0);
        glVertex3f(ind[i].x-1.0,ind[i].y,ind[i].z);
        glNormal3f(0.0,-1.0,1.0);
        glVertex3f(ind[i].x,ind[i].y-1.0,ind[i].z);
        glNormal3f(1.0,0.0,1.0);
        glVertex3f(ind[i].x+1.0,ind[i].y,ind[i].z);
      glEnd();
    }
    glDepthFunc(GL_LESS);
  }

  if (displayInd==2) {
    glMaterialfv(GL_FRONT,GL_AMBIENT,indAmbient);
    glMaterialfv(GL_FRONT,GL_DIFFUSE,indDiffuse);
    glMaterialfv(GL_FRONT,GL_SPECULAR,indSpecular);
    glMaterialf(GL_FRONT,GL_SHININESS,indShininess);
    glMaterialfv(GL_FRONT,GL_EMISSION,indEmissive);
    ind=env->individuals;
    for (i=0;i<env->nInd;i++) {
      glPushMatrix();
      glTranslatef(ind[i].x,ind[i].y,ind[i].z+0.05);
      glScalef(1.0,1.0,5.0/(xSize*0.2));
      glutSolidSphere(1.0,7,7);
      glPopMatrix();
    }
  }

  glutSwapBuffers();
}

/*
 * reshape display
 * ---------------
 * parameters
 *   int w: width
 *   int h: height
 */

void
reshape(int w,int h)
{
  windowWidth=w;
  windowHeight=h;
  if (windowWidth<grabX+grabWidth)
    grabWidth=windowWidth-grabX;
  if (windowHeight<grabY+grabHeight)
    grabHeight=windowHeight-grabY;
  glViewport(0,0,w,h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective(50.0,1.0*(GLfloat)windowWidth/(GLfloat)windowHeight,1.0,30.0);
  glMatrixMode(GL_MODELVIEW);
}

/*
 * idle function
 * -------------
 * parameters
 *   void: nothing
 * returns
 *   void: nothing
 */

void
idle(void)
{
  if (env->startSequence) {
    env->frameSkipCounter--;
    if (env->frameSkipCounter==0) {
      env->frameSkipCounter=env->frameSkip;
      saveResults(env);
    }
    if (env->landscapeNr==env->nSequence)
      env->startSequence=0;
  }
  if (env->running) {
    moveIndividuals(env->landscape,env->xSize,env->ySize,
                    env->individuals,env->nInd);
    envInteraction(env->landscape,env->xSize,env->ySize,env->staticLandscape,
                   env->individuals,env->nInd);
    placeIndividualsOnLandscape(env->landscape,env->xSize,env->ySize,
                                env->individuals,env->nInd);
    if (env->output) {
      glutPostRedisplay();
    }
  }
}

void
distributeHills(void)
{
  initLandscape(env->staticLandscape,env->xSize,env->ySize,
                env->nHills,gaussian);
  initIndPos(env);
  copyLandscape(env->landscape,env->xSize,env->ySize,
                env->staticLandscape);
  placeIndividualsOnLandscape(env->landscape,env->xSize,env->ySize,
                              env->individuals,env->nInd);
}

/*
 * handle special keys
 * -------------------
 * parameters
 *   unsigned char key: pressed key
 *   int x:             x coordinate of mouse
 *   int y:             x coordinate of mouse
 * returns
 *   void:              nothing
 */

void
specialKey(int key,int x,int y)
{
  switch (key) {
  case GLUT_KEY_LEFT:  angle1-=10.0; glutPostRedisplay(); break;
  case GLUT_KEY_RIGHT: angle1+=10.0; glutPostRedisplay(); break;
  case GLUT_KEY_UP:    angle2-=10.0; glutPostRedisplay(); break;
  case GLUT_KEY_DOWN:  angle2+=10.0; glutPostRedisplay(); break;
  default: break;
  }
}

/*
 * handle user input
 * -----------------
 * parameters
 *   unsigned char key: pressed key
 *   int x:             x coordinate of mouse
 *   int y:             x coordinate of mouse
 * returns
 *   void:              nothing
 */

void
key(unsigned char c,int x,int y)
{
  int  selection;
  char filename[256];

  if (inputMode) {
    if (c=='\b') {
      if (inputPos>0) {
        inputPos--;
        inputBuffer[inputPos]=0;
        glutPostRedisplay();
      }
      return;
    }
    if (c==13) {
      inputMode=0;
      switch (inputType) {
        case INPUT_INT:   sscanf(inputBuffer,"%d",(int *)inputPntr);  break;
        case INPUT_FLOAT: sscanf(inputBuffer,"%f",(float *)inputPntr);break;
        default:          break;
      }
      if (inputFunc!=NULL)
        inputFunc();
      glutPostRedisplay();
      return;
    }
    inputBuffer[inputPos++]=c;
    inputBuffer[inputPos]=0;
    glutPostRedisplay();
    if (inputPos>255)
      inputPos=255;
    return;
  }

  selection=c-'0';
  if (selection>=1 && selection<=maxSelection && env->started==0) {
    setParameters(env,menu,selection);
    initLandscape(env->staticLandscape,env->xSize,env->ySize,
                  env->nHills,gaussian);
    initIndPos(env);
    copyLandscape(env->landscape,env->xSize,env->ySize,env->staticLandscape);
    glutPostRedisplay();
    return;
  }

  if (c>='a' && c<='z') {
    c-='a';
    c+='A';
  }

  switch (c) {
  case 'O': env->output=!env->output; break;
  case 'M': displayMenu=!displayMenu;glutPostRedisplay();break;
  case 'T': getNextSelection(&displayInd,MAX_NINDDIS);break;
  case 'U': displayLines=!displayLines;glutPostRedisplay();break;
  case 'I': getInputInt(c,&env->nInd,NULL); break;
  case 'H': if (!env->started)
              getInputInt(c,&env->nHills,distributeHills);break;
  case 'D': getNextSelection(&env->shape,MAX_NSHAPES);break;
  case 'X': getInputInt(c,&env->xSize,NULL);break;
  case 'Y': getInputInt(c,&env->xSize,NULL);break;
  case 'G': getInputFloat(c,&env->sigma,initShapes);break;
  case 'V': getNextSelection(&env->updateVelocity,MAX_UDV);break;
  case 'N': getInputFloat(c,&env->noise,NULL);break;
  case 'L': getInputInt(c,&env->latency,NULL);break;
  case 'E': env->running=0;
            env->started=0;
            env->startSequence=0;
            glutIdleFunc(NULL);
            break;
  case 'S': env->running=0;
            glutIdleFunc(NULL);
            break;
  case 'C': if (env->started) {
              env->running=1;
              glutIdleFunc(idle);
            }
            break;
  case 'P': if (env->started) {
              saveResults(env);
            }
            break;
  case 'K': getInputInt(c,&env->nSequence,NULL);break;
  case 'W': if (env->started) {
              env->startSequence=1;
              env->landscapeNr=0;
              env->frameSkipCounter=1;
            }
            break;
  case 'F': getInputInt(c,&env->frameSkip,NULL);break;
  case 'Z': env->startSequence=1;
            env->landscapeNr=0;
            env->frameSkipCounter=1;
  case 'R': env->running=1;
            env->started=1;
            initIndividuals(env->individuals,env->nInd,env->updateVelocity,
              env->noise,env->latency+1,env->landscape,env->xSize,env->ySize,
              env->shape);
            envInteraction(env->landscape,env->xSize,env->ySize,
                           env->staticLandscape,env->individuals,env->nInd);
            placeIndividualsOnLandscape(env->landscape,env->xSize,env->ySize,
                                        env->individuals,env->nInd);
            glutPostRedisplay();
            glutIdleFunc(idle);
            break;
  case 'Q': exit(0);
  case '+': sprintf(filename,"%sstatLandscape.dat",env->filename);
            writeLandFile(filename,env->staticLandscape,env->xSize,env->ySize);
            sprintf(filename,"%sstatLandscape.ppm",env->filename);
            savePPMLandscape(env,filename);
	    break;
  case '-': sprintf(filename,"%sstatLandscape.dat",env->filename);
            readLandFile(filename,env->staticLandscape,env->xSize,env->ySize);
            copyLandscape(env->landscape,env->xSize,env->ySize,
                          env->staticLandscape);
            glutPostRedisplay();
	    break;
  default:  displayMenu=1;
            glutPostRedisplay();
            break;
  }
}

/*
 * init open gl
 * ------------
 * parameters
 *   void: nothing
 * returns
 *   void: nothing
 */

void
init(void)
{
  glClearColor(0.0,0.0,0.0,0.0);
  glShadeModel(GL_SMOOTH);
  glEnable(GL_NORMALIZE);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_DEPTH_TEST);
}

/*
 * beginning of main program
 * -------------------------
 * parameters
 *   int argc:    number of arguments
 *   char **argv: argument vector
 * returns
 *   int:         error flag
 */

int
main(int argc,char **argv)
{
  int   nHills=50;

  srand48(1);
  createEnvironment(101,101,nHills,0.01);
  glutInit(&argc,argv);
  glutInitDisplayMode(GLUT_DOUBLE|GLUT_RGB|GLUT_DEPTH);
  glutInitWindowSize(windowWidth,windowHeight);
  glutCreateWindow("Landscape");
  init();
  glutReshapeFunc(reshape);
  glutDisplayFunc(display);
  glutKeyboardFunc(key);
  glutSpecialFunc(specialKey);
  glutMainLoop();
  destroyEnvironment(env);
  return 0;
}







